<?php

/**
 * ---------------------------------------------------------------------
 *
 * GLPI - Gestionnaire Libre de Parc Informatique
 *
 * http://glpi-project.org
 *
 * @copyright 2015-2025 Teclib' and contributors.
 * @licence   https://www.gnu.org/licenses/gpl-3.0.html
 *
 * ---------------------------------------------------------------------
 *
 * LICENSE
 *
 * This file is part of GLPI.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * ---------------------------------------------------------------------
 */

use function Safe\preg_replace;

/**
 * Update from 9.4.2 to 9.4.3
 *
 * @return bool
 **/
function update942to943()
{
    /**
     * @var DBmysql $DB
     * @var Migration $migration
     */
    global $DB, $migration;

    $updateresult     = true;

    $migration->setVersion('9.4.3');

    /** Fix URL of images inside ITIL objects contents */
    // This is an exact copy of the same process used in "update940to941()" and "update941to942()"
    // which was not working for elements having a simple quote in their content.
    // It has been fixed there for people who had not yet updated to 9.4.1 / 9.4.2 but have to
    // be put back here for people already having updated to 9.4.1 / 9.4.2.
    $migration->displayMessage(__('Fix URL of images in ITIL tasks, followups and solutions.'));

    // Search for contents that does not contains the itil object parameter after the docid parameter
    // (i.e. having a quote that ends the href just after the docid param value).
    // 1st capturing group is the end of href attribute value
    // 2nd capturing group is the href attribute ending quote
    $quotes_possible_exp   = ['\'', '&apos;', '&#39;', '&#x27;', '"', '&quot', '&#34;', '&#x22;'];
    $missing_param_pattern = '(document\.send\.php\?docid=[0-9]+)(' . implode('|', $quotes_possible_exp) . ')';

    $itil_mappings = [
        'Change' => [
            'itil_table' => 'glpi_changes',
            'itil_fkey'  => 'changes_id',
            'task_table' => 'glpi_changetasks',
        ],
        'Problem' => [
            'itil_table' => 'glpi_problems',
            'itil_fkey'  => 'problems_id',
            'task_table' => 'glpi_problemtasks',
        ],
        'Ticket' => [
            'itil_table' => 'glpi_tickets',
            'itil_fkey'  => 'tickets_id',
            'task_table' => 'glpi_tickettasks',
        ],
    ];

    $fix_content_fct = (fn($content, $itil_id, $itil_fkey)
        // Add itil object param between docid param ($1) and ending quote ($2)
        => preg_replace(
            '/' . $missing_param_pattern . '/',
            '$1&amp;' . http_build_query([$itil_fkey => $itil_id]) . '$2',
            $content
        ));

    foreach ($itil_mappings as $itil_type => $itil_specs) {
        $itil_fkey  = $itil_specs['itil_fkey'];
        $task_table = $itil_specs['task_table'];

        // Fix followups and solutions
        foreach (['glpi_itilfollowups', 'glpi_itilsolutions'] as $itil_element_table) {
            $elements_to_fix = $DB->request(
                [
                    'SELECT'    => ['id', 'items_id', 'content'],
                    'FROM'      => $itil_element_table,
                    'WHERE'     => [
                        'itemtype' => $itil_type,
                        'content'  => ['REGEXP', $missing_param_pattern],
                    ],
                ]
            );
            foreach ($elements_to_fix as $data) {
                $data['content'] = $fix_content_fct($data['content'], $data['items_id'], $itil_fkey);
                $DB->update($itil_element_table, $data, ['id' => $data['id']]);
            }
        }

        // Fix tasks
        $tasks_to_fix = $DB->request(
            [
                'SELECT'    => ['id', $itil_fkey, 'content'],
                'FROM'      => $task_table,
                'WHERE'     => [
                    'content'  => ['REGEXP', $missing_param_pattern],
                ],
            ]
        );
        foreach ($tasks_to_fix as $data) {
            $data['content'] = $fix_content_fct($data['content'], $data[$itil_fkey], $itil_fkey);
            $DB->update($task_table, $data, ['id' => $data['id']]);
        }
    }
    /** /Fix URL of images inside ITIL objects contents */

    // add is_private field to change and problems
    $migration->addField('glpi_changetasks', 'is_private', 'bool');
    $migration->addField('glpi_problemtasks', 'is_private', 'bool');
    $migration->addKey('glpi_changetasks', 'is_private');
    $migration->addKey('glpi_problemtasks', 'is_private');

    /** Crontask missing from fresh install */
    $migration->addCrontask(
        'PurgeLogs',
        'PurgeLogs',
        7 * DAY_TIMESTAMP,
        param: 24,
    );
    /** /Crontask missing from fresh install */

    // ************ Keep it at the end **************
    $migration->executeMigration();

    return $updateresult;
}
